<?php

namespace Mnv\Core\Filesystem;

use ErrorException;
use FilesystemIterator;
use Mnv\Core\Collections\Contracts\Filesystem\FileNotFoundException;
use Mnv\Core\Collections\Traits\Macroable;
use RuntimeException;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Mime\MimeTypes;

class Filesystem
{
    use Macroable;



    /** @var string  */
    public string $realPath;

    /** @var string|array|string[]  */
    public string $path;


    /** @var array  */
    public array $response = [];

    /** Images */
    public array $ext_img = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp'];
    public array $ext_copy_img = ['svg', 'ico'];

    /** Files */
    public array $ext_file = ['doc', 'docx', 'rtf', 'pdf', 'xls', 'xlsx', 'txt', 'csv', 'ppt', 'pptx', 'epub'];

    /** Video */
    public array $ext_video = ['mov', 'mpeg', 'm4v', 'mp4', 'avi', 'mpg', 'wma', "webm"];

    /** Audio */
    public array $ext_music = ['mp3', 'mpga', 'm4a', 'ac3', 'aiff', 'mid', 'ogg', 'wav'];

    /** Archives */
    public array $ext_misc  = ['zip', 'rar', 'gz', 'tar', 'dmg'];

    /** @var string[] */
    public static $_hidden_image_folders = ['admin', 'large', 'medium', 'small'];

    /** @var string[] */
    public static $_hidden_folders = ['vendor', 'large', 'medium', 'small'];

    /** @var string[] */
    protected array $sizes = ['large', 'medium', 'small'];

    /**
     * Определите, существует ли файл или каталог.
     *
     * @param string $path
     * @return bool
     */
    public function exists(string $path): bool
    {
        return file_exists($path);
    }

    /**
     *Определите, отсутствует ли файл или каталог.
     *
     * @param  string  $path
     * @return bool
     */
    public function missing(string $path): bool
    {
        return ! $this->exists($path);
    }

    /**
     * Получить содержимое файла.
     *
     * @param  string  $path
     * @param  bool  $lock
     * @return string
     *
     * @throws \Mnv\Core\Collections\Contracts\Filesystem\FileNotFoundException
     */
    public function get(string $path, bool $lock = false): string
    {
        if ($this->isFile($path)) {
            return $lock ? $this->sharedGet($path) : file_get_contents($path);
        }

        throw new FileNotFoundException("File does not exist at path {$path}.");
    }

    /**
     * Получить содержимое файла с общим доступом.
     *
     * @param  string  $path
     * @return string
     */
    public function sharedGet(string $path): string
    {
        $contents = '';

        $handle = fopen($path, 'rb');

        if ($handle) {
            try {
                if (flock($handle, LOCK_SH)) {
                    clearstatcache(true, $path);

                    $contents = fread($handle, $this->size($path) ?: 1);

                    flock($handle, LOCK_UN);
                }
            } finally {
                fclose($handle);
            }
        }

        return $contents;
    }

    /**
     * Получите возвращаемое значение файла.
     *
     * @param  string  $path
     * @return mixed
     *
     * @throws \Mnv\Core\Collections\Contracts\Filesystem\FileNotFoundException
     */
    public function getRequire(string $path)
    {
        if ($this->isFile($path)) {
            return require $path;
        }

        throw new FileNotFoundException("File does not exist at path {$path}.");
    }

    /**
     * Затребуйте данный файл один раз.
     *
     * @param  string  $file
     * @return mixed
     */
    public function requireOnce(string $file)
    {
        require_once $file;
    }

    /**
     * Получите MD5-хэш файла по заданному пути.
     *
     * @param  string  $path
     * @return string
     */
    public function hash(string $path)
    {
        return md5_file($path);
    }

    /**
     * Запишите содержимое файла.
     *
     * @param  string  $path
     * @param  string  $contents
     * @param  bool  $lock
     * @return int|bool
     */
    public function put(string $path, string $contents, bool $lock = false)
    {
        return file_put_contents($path, $contents, $lock ? LOCK_EX : 0);
    }

    /**
     * Запишите содержимое файла, заменив его атомарно, если он уже существует.
     *
     * @param  string  $path
     * @param  string  $content
     * @return void
     */
    public function replace(string $path, string $content)
    {
        // If the path already exists and is a symlink, get the real path...
        clearstatcache(true, $path);

        $path = realpath($path) ?: $path;

        $tempPath = tempnam(dirname($path), basename($path));

        // Fix permissions of tempPath because `tempnam()` creates it with permissions set to 0600...
        chmod($tempPath, 0777 - umask());

        file_put_contents($tempPath, $content);

        rename($tempPath, $path);
    }

    /**
     *Добавить в файл.
     *
     * @param  string  $path
     * @param  string  $data
     * @return int
     */
    public function prepend(string $path, string $data)
    {
        if ($this->exists($path)) {
            return $this->put($path, $data . $this->get($path));
        }

        return $this->put($path, $data);
    }

    /**
     * Добавить в файл.
     *
     * @param  string  $path
     * @param  string  $data
     * @return int
     */
    public function append(string $path, string $data)
    {
        return file_put_contents($path, $data, FILE_APPEND);
    }

    /**
     * Получите или установите режим UNIX для файла или каталога.
     *
     * @param  string  $path
     * @param  int|null  $mode
     * @return mixed
     */
    public function chmod(string $path, $mode = null)
    {
        if ($mode) {
            return chmod($path, $mode);
        }

        return substr(sprintf('%o', fileperms($path)), -4);
    }

    /**
     * Удалите файл по заданному пути.
     *
     * @param  string|array  $paths
     * @return bool
     */
    public function delete($paths)
    {
        $paths = is_array($paths) ? $paths : func_get_args();

        $success = true;

        foreach ($paths as $path) {
            try {
                if (! @unlink($path)) {
                    $success = false;
                }
            } catch (ErrorException $e) {
                $success = false;
            }
        }

        return $success;
    }

    /**
     * Переместите файл в новое местоположение.
     *
     * @param  string  $path
     * @param  string  $target
     * @return bool
     */
    public function move(string $path, string $target)
    {
        return rename($path, $target);
    }

    /**
     * Скопируйте файл в новое местоположение.
     *
     * @param  string  $path
     * @param  string  $target
     * @return bool
     */
    public function copy(string $path, string $target)
    {
        return copy($path, $target);
    }

    /**
     * Создайте символическую ссылку на целевой файл или каталог.
     * В Windows создается жесткая ссылка, если целью является файл.
     *
     * @param  string  $target
     * @param  string  $link
     * @return void
     */
    public function link(string $target, string $link)
    {
        if (! windows_os()) {
            return symlink($target, $link);
        }

        $mode = $this->isDirectory($target) ? 'J' : 'H';

        exec("mklink /{$mode} " . escapeshellarg($link) . ' ' . escapeshellarg($target));
    }

    /**
     * Извлеките имя файла из пути к файлу.
     *
     * @param string $path
     * @return string
     */
    public function name(string $path): string
    {
        return pathinfo($path, PATHINFO_FILENAME);
    }

    /**
     * Extract the trailing name component from a file path.
     *
     * @param  string  $path
     * @return string
     */
    public function basename(string $path): string
    {
        return pathinfo($path, PATHINFO_BASENAME);
    }

    /**
     * Извлеките родительский каталог из пути к файлу.
     *
     * @param  string  $path
     * @return string
     */
    public function dirname(string $path): string
    {
        return pathinfo($path, PATHINFO_DIRNAME);
    }

    /**
     * Извлеките расширение файла из пути к файлу.
     *
     * @param  string  $path
     * @return string
     */
    public function extension(string $path): string
    {
        return pathinfo($path, PATHINFO_EXTENSION);
    }

    /**
     * Угадайте расширение файла по mime-типу данного файла.
     *
     * @param  string  $path
     * @return string|null
     */
    public function guessExtension(string $path): ?string
    {
        if (! class_exists(MimeTypes::class)) {
            throw new RuntimeException(
                'To enable support for guessing extensions, please install the symfony/mime package.'
            );
        }

        return (new MimeTypes)->getExtensions($this->mimeType($path))[0] ?? null;
    }

    /**
     * Получите тип файла для данного файла.
     *
     * @param  string  $path
     * @return string
     */
    public function type(string $path): string
    {
        return filetype($path);
    }

    /**
     * Получите mime-тип данного файла.
     *
     * @param  string  $path
     * @return string|false
     */
    public function mimeType(string $path)
    {
        return finfo_file(finfo_open(FILEINFO_MIME_TYPE), $path);
    }

    /**
     * Получите размер файла для данного файла.
     *
     * @param  string  $path
     * @return int
     */
    public function size(string $path): int
    {
        return filesize($path);
    }

    /**
     * Получите время последнего изменения файла.
     *
     * @param  string  $path
     * @return int
     */
    public function lastModified(string $path): int
    {
        return filemtime($path);
    }

    /**
     * Определите, является ли указанный путь каталогом.
     *
     * @param  string  $directory
     * @return bool
     */
    public function isDirectory(string $directory): bool
    {
        return is_dir($directory);
    }

    /**
     * Определите, доступен ли данный путь для чтения.
     *
     * @param  string  $path
     * @return bool
     */
    public function isReadable(string $path): bool
    {
        return is_readable($path);
    }

    /**
     * Определите, доступен ли данный путь для записи.
     *
     * @param  string  $path
     * @return bool
     */
    public function isWritable(string $path): bool
    {
        return is_writable($path);
    }

    /**
     * Определите, является ли указанный путь файлом.
     *
     * @param  string  $file
     * @return bool
     */
    public function isFile(string $file): bool
    {
        return is_file($file);
    }

    /**
     * Найдите имена путей, соответствующие заданному шаблону.
     *
     * @param  string  $pattern
     * @param  int  $flags
     * @return array
     */
    public function glob(string $pattern, $flags = 0): array
    {
        return glob($pattern, $flags);
    }

    /**
     * Получите массив всех файлов в каталоге.
     *
     * @param  string  $directory
     * @param  bool  $hidden
     * @return \Symfony\Component\Finder\SplFileInfo[]
     */
    public function files(string $directory, bool $hidden = false): array
    {
        return iterator_to_array(Finder::create()->files()->ignoreDotFiles(! $hidden)->in($directory)->depth(0)->sortByName(), false);
    }

    /**
     * Получите все файлы из заданного каталога (рекурсивно).
     *
     * @param  string  $directory
     * @param  bool  $hidden
     * @return \Symfony\Component\Finder\SplFileInfo[]
     */
    public function allFiles(string $directory, bool $hidden = false): array
    {
        return iterator_to_array(Finder::create()->files()->ignoreDotFiles(! $hidden)->in($directory)->sortByName(), false);
    }

    /**
     * Получите все каталоги в пределах данного каталога.
     *
     * @param  string  $directory
     * @return array
     */
    public function directories(string $directory): array
    {
        $directories = [];

        foreach (Finder::create()->in($directory)->directories()->depth(0)->sortByName() as $dir) {
            $directories[] = $dir->getPathname();
        }

        return $directories;
    }

    /**
     * Убедитесь, что каталог существует.
     *
     * @param  string  $path
     * @param  int  $mode
     * @param  bool  $recursive
     * @return void
     */
    public function ensureDirectoryExists(string $path, $mode = 0755, $recursive = true)
    {
        if (! $this->isDirectory($path)) {
            $this->makeDirectory($path, $mode, $recursive);
        }
    }

    /**
     * Создайте каталог.
     *
     * @param  string  $path
     * @param  int  $mode
     * @param  bool  $recursive
     * @param  bool  $force
     * @return bool
     */
    public function makeDirectory(string $path, $mode = 0755, $recursive = false, $force = false): bool
    {
        if ($force) {
            return @mkdir($path, $mode, $recursive);
        }

        return mkdir($path, $mode, $recursive);
    }

    /**
     * Переместите каталог.
     *
     * @param  string  $from
     * @param  string  $to
     * @param  bool  $overwrite
     * @return bool
     */
    public function moveDirectory(string $from, string $to, $overwrite = false)
    {
        if ($overwrite && $this->isDirectory($to) && ! $this->deleteDirectory($to)) {
            return false;
        }

        return @rename($from, $to) === true;
    }

    /**
     * Скопируйте каталог из одного места в другое.
     *
     * @param  string  $directory
     * @param  string  $destination
     * @param  int|null  $options
     * @return bool
     */
    public function copyDirectory(string $directory, string $destination, $options = null)
    {
        if (! $this->isDirectory($directory)) {
            return false;
        }

        $options = $options ?: FilesystemIterator::SKIP_DOTS;

        // If the destination directory does not actually exist, we will go ahead and
        // create it recursively, which just gets the destination prepared to copy
        // the files over. Once we make the directory we'll proceed the copying.
        $this->ensureDirectoryExists($destination, 0777);

        $items = new FilesystemIterator($directory, $options);

        foreach ($items as $item) {
            // As we spin through items, we will check to see if the current file is actually
            // a directory or a file. When it is actually a directory we will need to call
            // back into this function recursively to keep copying these nested folders.
            $target = $destination.'/'.$item->getBasename();

            if ($item->isDir()) {
                $path = $item->getPathname();

                if (! $this->copyDirectory($path, $target, $options)) {
                    return false;
                }
            }

            // If the current items is just a regular file, we will just copy this to the new
            // location and keep looping. If for some reason the copy fails we'll bail out
            // and return false, so the developer is aware that the copy process failed.
            else {
                if (! $this->copy($item->getPathname(), $target)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Рекурсивно удалите каталог.
     *
     * The directory itself may be optionally preserved.
     *
     * @param  string  $directory
     * @param  bool  $preserve
     * @return bool
     */
    public function deleteDirectory(string $directory, $preserve = false)
    {
        if (! $this->isDirectory($directory)) {
            return false;
        }

        $items = new FilesystemIterator($directory);

        foreach ($items as $item) {
            // If the item is a directory, we can just recurse into the function and
            // delete that sub-directory otherwise we'll just delete the file and
            // keep iterating through each file until the directory is cleaned.
            if ($item->isDir() && ! $item->isLink()) {
                $this->deleteDirectory($item->getPathname());
            }

            // If the item is just a file, we can go ahead and delete it since we're
            // just looping through and waxing all of the files in this directory
            // and calling directories recursively, so we delete the real path.
            else {
                $this->delete($item->getPathname());
            }
        }

        if (! $preserve) {
            @rmdir($directory);
        }

        return true;
    }

    /**
     * Удалите все каталоги в пределах данного каталога.
     *
     * @param  string  $directory
     * @return bool
     */
    public function deleteDirectories(string $directory)
    {
        $allDirectories = $this->directories($directory);

        if (! empty($allDirectories)) {
            foreach ($allDirectories as $directoryName) {
                $this->deleteDirectory($directoryName);
            }

            return true;
        }

        return false;
    }

    /**
     * Очистите указанный каталог от всех файлов и папок.
     *
     * @param  string  $directory
     * @return bool
     */
    public function cleanDirectory(string $directory)
    {
        return $this->deleteDirectory($directory, true);
    }


    /**
     *
     * @param int $size
     * @return string
     */
    public function formattedFileSize(int $size): string
    {
        if (!$size OR $size < 1) return '0 b';

        $prefix = array("b", "Kb", "Mb", "Gb", "Tb");
        $exp = floor(log($size, 1024)) | 0;

        return round($size / (pow(1024, $exp)), 2).' '.$prefix[$exp];
    }

}
