<?php

namespace Mnv\Core\Filesystem;

/**
 * Class FilesystemManager
 * @package Mnv\Core\Filesystem
 */
class FilesystemManager extends Filesystem
{

    /** @var string */
    protected string $table = 'files';

    /** @var string  */
    protected string $primaryKey = 'fileId';

    /** @var string  */
    protected string $orderBy = 'fileId DESC';

    /**
     * FileManager constructor.
     * @param string|null $path
     */
    public function __construct(string $directory = null, string $path = null)
    {
        $this->ensureDirectoryExists(GLOBAL_ROOT . $directory . DIRECTORY_SEPARATOR . $path);

        $this->realPath = realpath(GLOBAL_ROOT . $directory . DIRECTORY_SEPARATOR . $path) . DIRECTORY_SEPARATOR;
        if (strpos($this->realPath, GLOBAL_ROOT . $directory . DIRECTORY_SEPARATOR) === false) {
            $this->realPath = GLOBAL_ROOT . $directory . DIRECTORY_SEPARATOR;
        }

        $this->path = str_replace(GLOBAL_ROOT . $directory . DIRECTORY_SEPARATOR, '', $this->realPath);
    }

    /**
     * Получить все папки по заданному пути
     * @param array $hidden
     * @param bool|false $isSize
     * @return array
     */
    public function getDirectories(array $hidden, bool $isSize = false): array
    {
        $list = [];
        $size = 0;
        $directories = $this->directories($this->realPath);

        foreach ($directories as $directory) {
            $directoryBaseName  = $this->basename($directory);
            $directoryName      = $this->name($directory);
            $directoryDirName   = $this->dirname($directory);

            if (!in_array($directoryName, $hidden)) {
                if ($isSize) list($size, $files_count, $folders_count) = $this->folder_info($directory);
                $list[] = [
                    'is_dir' => true,
                    'dirname'   => $directoryDirName,
                    'path'   => $directoryBaseName,
                    'name'   => $directoryName,
                    'mtime'  => is_writable($directory) ? $this->lastModified($directory) : '',
                    'size'   => $this->formattedFileSize($size),
//                    'size'  => $size,
                ];
            }
        }

        return $list;
    }


    private function sorting($query, $filters)
    {
        if (!empty($query)) {
            connect()->like('fileName', "%$query%");
        }
        if (!empty($filters)) {
            $filterMap = ['date' => 'addedOn ASC', 'size' => 'size ASC', 'name' => 'fileName ASC'];
            connect()->orderBy($filterMap[$filters] ?? $this->orderBy);
        } else {
            connect()->orderBy($this->orderBy);
        }
    }

    public function getTotalPage(?array $where, ?string $query, ?string $filters, ?string $type, int $managerId, int $limit, int $page)
    {
        if ($type === 'mine') {
            $where['addedBy'] = $managerId;
        } elseif (!empty($type)) {
            connect()->where('mimeType', $type);
        }
        $this->sorting($query, $filters);

        $list = connect($this->table)->count('*', 'count')->where($where)->get('array');

        if ($list['count'] > 0) {
            $pagination = getPageNums($list['count'], $page, $limit, 0, 4, 4, 9, null, false);
            unset(
                $pagination['pages'],
                $pagination['lastPage'],
                $pagination['nextPage'],
                $pagination['firstPage'],
                $pagination['previousPage'],
                $pagination['startIteration']
            );

        } else {
            $pagination['currentPage'] = 1;
            $pagination['endIteration'] = 0;
            $pagination['totalItems'] = 0;
            $pagination['totalPages'] = 0;
        }

        return $pagination;
    }


    /**
     * Список файлов
     *
     * @param array|null $where
     * @param string|null $query
     * @param string|null $filters
     * @param string|null $type
     * @param int $managerId
     * @param int $limit
     * @param int $page
     * @return array|mixed
     */
    public function all(?array $where, ?string $query, ?string $filters, ?string $type, int $managerId, int $limit, int $page)
    {
        if ($type === 'mine') {
            $where['addedBy'] = $managerId;
        } elseif (!empty($type)) {
            connect()->where('mimeType', $type);
        }

        $this->sorting($query, $filters);
        $files = connect($this->table)->where($where)->pagination($limit, $page)->getAll('array');
        return collect($files)->map(function ($file)  {
            $extension = strtolower($this->extension($this->realPath . $file['fileName']));

            return [
                'is_dir'    => false,
                'fileId'    => $file['fileId'],
                'fileName'  => $file['fileName'],
                'fileSize'  => $file['size'],
                'fileMime'  => $file['mimeType'],
                'fileType'  => $this->getFileType($extension),
                'size'      => $this->formattedFileSize($file['size']),
                'mtime'     => is_writable($this->realPath . $file['fileName']) ? $this->lastModified($this->realPath . $file['fileName']) : '',
                'icon'      => GLOBAL_URL . '/admin/assets/ico/' . $extension . '.svg',
                'original'  => GLOBAL_URL . $file['path'] . $file['fileName'],
                'url'       => GLOBAL_URL . $file['path'] . $file['fileName'],

            ];
        })->all();
    }

    /**
     * Получение файла
     *
     * @param int $fileId
     * @return array
     */
    public function read(int $fileId): array
    {
        $file = connect($this->table)->where($this->primaryKey, $fileId)->get('array');

        if ($file) {
            $extension = strtolower($this->extension($this->realPath . $file['fileName']));

            return [
                'fileId'    => $file['fileId'],
                'extension' => $extension,
                'fileName'  => $file['fileName'],
                'fileSize'  => $file['size'],
                'fileType'  => $this->getFileType($extension),
                'size'      => $this->formattedFileSize($file['size']),
                'url'       => GLOBAL_URL . $file['path'] . $file['fileName'],
                'icon'      => GLOBAL_URL . '/admin/assets/ico/' . $extension . '.svg',
            ];
        }

        return [];
    }

    private function getFileType(string $extension): string
    {
        if (in_array($extension, $this->ext_video)) {
            return 'video';
        } elseif (in_array($extension, $this->ext_img)) {
            return 'image';
        } elseif (in_array($extension, $this->ext_copy_img)) {
            return 'svg';
        } elseif (in_array($extension, $this->ext_music)) {
            return 'audio';
        } elseif (in_array($extension, $this->ext_misc)) {
            return 'archive';
        } else {
            return 'unknown';
        }
    }


    /**
     * Определение размера каталога / Determine directory size
     * determineDirectorySize
     *
     * @param string $path
     * @return array
     * folder_info
     */
    private function folder_info(string $path): array
    {
        $total_size = 0;
        $files_count = 0;
        $folders_count = 0;

        if ($this->exists($path)) {
            foreach ($this->directories($path) as $directory) {
                if (!in_array($this->name($directory), self::$_hidden_image_folders)) {
                    [$size] = $this->folder_info($directory);
                    $total_size += $size;
                    $folders_count++;
                }
            }

            foreach ($this->files($path, false) as $file) {
                $total_size += $this->size($file->getRealPath());
                $files_count++;
            }
        }

        return [$total_size, $files_count, $folders_count];
    }

    /**
     * Удаление файла / файлов
     *
     * @param array|int $fileIds
     * @return bool
     */
    public function deleteFiles($fileIds): bool
    {
        $files = connect($this->table)->in($this->primaryKey, $fileIds)->getAll('array');

        if (!$files) {
            return false;
        }
        collect($files)->each(function ($file) {
            $this->deleteFileWithSizes($file['path'], $file['fileName']);
            connect($this->table)->where($this->primaryKey, $file['fileId'])->delete();
        })->all();

        return true;
    }

    /**
     * php 8 > array_map(fn($size) => $path . $size . '/' . $fileName, $this->sizes)
     * @param  string  $path
     * @param  string  $fileName
     *
     * @return void
     */
    private function deleteFileWithSizes(string $path, string $fileName): void
    {
        $fullPaths = array_merge([$path . $fileName],
            collect($this->sizes)->map(function ($size)  use ($path, $fileName) {
                return $path . $size . '/' . $fileName;
            })->all()
        );

        collect($fullPaths)->each(function ($fullPath) {
            $this->deleteFileIfExists(GLOBAL_ROOT . $fullPath);
        })->all();
    }

    private function deleteFileIfExists(string $filePath): void
    {
        if ($this->exists($filePath)) {
            unlink($filePath);
        }
    }

    /**
     * Удаление всех файлов с указаной директорией
     *
     * @param string $directory
     * @return bool Возвращам true, если удаление прошло успешно, или false в противном случае
     */
    public function deleteAllFilesThisDirectory(string $directory): bool
    {
        $deleted = connect($this->table)->where('directory', $directory)->delete();

        return $deleted !== false;
    }
}