<?php


namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Mnv\Core\Filesystem\Filesystem;
use Mnv\Core\Utilities\GUID;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

/**
 * Class AbstractImagineGenerator
 * @package Mnv\Core\Uploads
 */
abstract class AbstractGenerator extends Filesystem
{

    /** @var array|string[]  */
    protected array $sizes = array('large', 'medium', 'small');

    /** @var string  */
    public string $realPath;

    /** @var string */
    public string $path;

    /** @var string  */
    public string $uuid;

    /** @var int */
    public int $managerId;

    /** @var SymfonyUploadedFile */
    protected SymfonyUploadedFile $file;

    /** @var string $fileName */
    protected string $fileName;

    /** @var string  */
    protected string $driver = 'gd';

    /** @var string  */
    protected string $extension;

    /** @var array $fontOptions */
    protected array $fontOptions = [];

    /** @var string  */
    protected string $fontWatermark = GLOBAL_ROOT . '/admin/assets/fonts/verdana.ttf';

    /**
     * AbstractImagineGenerator constructor.
     * @param string $realPath
     * @param string $path
     * @param string|null $uuid
     * @param int $managerId
     */
    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        $this->realPath     = $realPath;
        $this->path         = $path;
        $this->uuid         = $uuid;
        $this->managerId    = $managerId;

    }
    abstract public function init(SymfonyUploadedFile $file);

    /**
     * если включена конвертация файла в `webp` то нам нужно знать,
     * если это `gif` то нельзя конвертировать в `webp`,
     * для остальных расширений разрешена конвертация в `webp`
     */
    protected function generatorExtension(string $originalExtension): void
    {
        $this->extension = $originalExtension === 'gif' || ConfigManager::getValue('force_webp') == 0
            ? $originalExtension
            : 'webp';
    }

    protected function generateFileName(string $originalName)
    {
        if (ConfigManager::getValue('keep_original_image_name')) {
            $this->fileName = pregGetValue('/([^\/]+)\.[^\.]+$/', $originalName);
        }

        /** новое название файла uuid */
        $this->fileName = !empty($this->uuid) ? $this->uuid : GUID::Format(GUID::Create(), false, '-');
    }

    /**
     * сбор опций обработки `watermark` $this->fontOptions = [
     * 'Размер шрифта (size)'        => 'watermark_font',
     * 'Цвет шрифта (color)'         => 'watermarkColor',
     * 'Прозрачность шрифта (alpha)' => 'watermark_opacity',  // пока не работает
     * 'Поворот текста (angle)'      => 'watermark_rotate'
     * ];
     */
    protected function getFontOptions()
    {
        $this->fontOptions = [
            'size' => ConfigManager::getValue('watermark_font'),
            'color' => [
                'light' => ConfigManager::getValue('watermark_color_light'),
                'dark' => ConfigManager::getValue('watermark_color_dark')
            ],
            'alpha' => (int)ConfigManager::getValue('watermark_opacity'),
            'angle' => ConfigManager::getValue('watermark_rotate'),
            'margin' => 20,
        ];
    }


    /**
     * @param array $file
     * @return bool|int|string|null
     */
    public function saveFile(array $file)
    {
        if (empty($file)) {
            return false;
        }

        $insertFile = array_filter([
            'directory' => $this->basename($this->realPath),
            'path'      => '/uploads/' . $this->path,
            'fileName'  => $file['name'],
            'size'      => $file['size'],
            'mimeType'  => $file['mimeType'] ?? null,
            'addedBy'   => $this->managerId ?: 0,
            'addedOn'   => gmdate('Y-m-d H:i:s'),
        ]);

        if (!empty($insertFile)) {
            return connect('files')->insert($insertFile) ?: false;
        }

        return false;
    }

    /**
     * Формируем размеры для обработки файла
     * @param $size
     * @return array
     */
    protected function autoSize($size): array
    {
        if (strpos($size, 'x') !== false) {
            $sizes = explode("x", $size);
            $width = intval($sizes[0]);
            $height = (count($sizes) == 2) ? intval($sizes[1]) : $width;
        } else {
            $width = intval($size);
            $height = intval($size);
        }

        return [$width, $height];
    }

}