<?php

namespace Mnv\Core\Uploads;

use Mnv\Core\ConfigManager;
use Imagine\Image\Box;
use Imagine\Image\ImageInterface;
use Mnv\Core\Uploads\Exceptions\InvalidParamException;
use Symfony\Component\HttpFoundation\File\UploadedFile as SymfonyUploadedFile;

/**
 * Class ImagineGenerator
 * @package Mnv\Core\Uploads
 */
class ImagineGenerator extends AbstractGenerator implements IGenerator
{

    /** @var ImageInterface $image */
    private ImageInterface $image;

    /** @var array|false[]|int[]  */
    private array $options = [];

    /** @var string $tmp */
    private string $tmp;

    public function __construct(string $realPath, string $path, ?string $uuid, int $managerId)
    {
        parent::__construct($realPath, $path, $uuid, $managerId);
        $this->setDriver();
    }

    /**
     *
     * @param SymfonyUploadedFile $file
     * @return ImagineGenerator
     */
    public function init(SymfonyUploadedFile $file): ImagineGenerator
    {
        $this->file = $file;
        $this->generatorExtension($file->getClientOriginalExtension());
        /** новое название файла uuid */
        $this->generateFileName($file->getClientOriginalName());
        /** сбор опций обработки `watermark` */
        $this->getFontOptions();

        $this->options = $this->mergeImageOptions();

        /** получение временного файла */
        $this->tmp = $this->file->getPathname();
        /** достаем временный файл для обработки */
        $this->image = $this->openImage($this->tmp);

        return $this;
    }

    /**
     * @throws InvalidParamException
     */
    public function save()
    {
        $images = [
            'name'      => "{$this->fileName}.{$this->extension}",
            'size'      => $this->file->getSize() ?? null,
            'mimeType'  => ($this->extension == 'webp') ? 'image/webp' : $this->file->getClientMimeType(),
        ];

        $this->resizeAndSaveImage($this->image->getSize()->getWidth(), $this->image->getSize()->getHeight(), $this->realPath . $this->fileName . '.' . $this->extension);

        foreach ($this->sizes as $size) {
            $this->resizeAndSaveThumbnail($size);
        }

        return $this->saveFile($images);
    }

    /**
     * @throws InvalidParamException
     */
    private function resizeAndSaveImage(int $width, int $height, string $path): void
    {
        $image = ConfigManager::getValue('allow_watermark') && $this->extension !== 'gif'
            ? BaseImage::textBox($this->tmp, ConfigManager::getValue('watermark_text'), $this->fontWatermark, ConfigManager::getValue('watermark_position'), $this->fontOptions)
            : BaseImage::resize($this->tmp, $width, $height);

        $this->saveImage($image, $width, $height, $path);
    }

    private function saveImage(ImageInterface $image, int $width, int $height, string $path): void
    {
        try {
            $image->thumbnail(new Box($width, $height))->save($path, $this->options);
        } catch (InvalidParamException $e) {
            echo $e->getMessage();
        }
    }

    /**
     * @throws InvalidParamException
     */
    private function resizeAndSaveThumbnail(string $size): void
    {
        $sizes = $this->autoSize(ConfigManager::getValue("{$size}_thumb"));
        $this->resizeAndSaveImage($sizes[0], $sizes[1], $this->realPath . $size . '/' . $this->fileName . '.' . $this->extension);
    }

    private function mergeImageOptions(): array
    {
        try {
            return BaseImage::mergeOptions($this->extension, ConfigManager::getValue('thumbnail_quality'));
        } catch (InvalidParamException $e) {
            echo $e->getMessage();
            return [];
        }
    }


    private function openImage(string $path): ImageInterface
    {
        try {
            return BaseImage::open($path);
        } catch (InvalidParamException $e) {
            // Re-throw to ensure the error is handled elsewhere
            print $e->getMessage();
        }
    }

    /** Получить драйвер обработчика файла */
    private function setDriver()
    {
        $drivers = [
            'imagick'  => \Imagine\Imagick\Imagine::class,
            'gd'       => \Imagine\Gd\Imagine::class,
            'gmagick'  => \Imagine\Gmagick\Imagine::class,
        ];

        if (ConfigManager::getValue('image_library') != 2) {
            if (extension_loaded('imagick') && class_exists('Imagick'))	{
                $this->driver = 'imagick';

                if (!\Imagick::queryFormats('WEBP') && function_exists('imagewebp') && ConfigManager::getValue('image_library') != 1 ) {
                    $this->driver = 'gd';
                }
            }
        }

        if (array_key_exists($this->driver, $drivers)) {
            BaseImage::setImagine(new $drivers[$this->driver]());
        } else {
            throw new \RuntimeException("Unsupported image driver: {$this->driver}");
        }



    }

}