<?php

namespace Mnv\Models\Cars;

use Mnv\Core\Database\Throwable\DatabaseException;
use Mnv\Core\Mnv;
use Mnv\Core\Model;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

/**
 * Class Options
 * @package Mnv\Models\Cars
 */
class Options extends Model
{
    protected string $table = 'car_options';

    protected string $primaryKey = 'optionId';

    protected string $orderBy = 'orderBy ASC';


    public $optionIds = [];
    public $options = [];
    public $modelIds = [];

    public function __construct(Request $request)
    {

        $this->id          = $request->get('id');
        $this->data        = $request->get('features');
        $this->optionIds   = $request->get('optionIds');
        $this->modelIds   = $request->get('modelIds');
        /** options  */
        $this->options = $request->get('options');

    }


    private function sortBy()
    {
        if (!empty($this->filter['query'])) {
            connect()->grouped(function($q) {
                $q->like('name', "%" . $this->filter['query'] . "%");
            });
        }
        if (!empty($this->filter['status'])) {
            connect()->where('status', $this->filter['status']);
        }

        connect()->whereNull('parentId',  false);

    }


    public function all($limit, $page)
    {
        $this->sortBy();

        return parent::all($limit, $page);
    }
    public function total(): void
    {
        $this->sortBy();

        parent::total();
    }


    public function prepare(array $data, int $managerId): bool
    {

        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');

        if (empty($this->id)) {

            $data['addedBy'] = $managerId;
            $data['addedOn'] = gmdate('Y-m-d H:i:s');
            $data['orderBy'] = connect()->table($this->table)->max('orderBy')->getValue() + 1;
            if ($this->id = $this->insert($data)) {
                return true;
            }
        }
        else {

            return $this->update($data);
        }

        return false;
    }


    /**
     * Удаление
     *
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     * @throws DatabaseException
     */
    public function remove(): bool
    {
        /** удаление свойства */
        if (parent::remove()) {
            /** удаление свойства у комплектации в связывающей таблице `car_options_value` */
            connect('car_option_models')->where($this->primaryKey, $this->id)->delete();
            connect('car_options_value')->where($this->primaryKey, $this->id)->delete();
            $optionIds = connect($this->table)->where('parentId', $this->id)->pluck($this->primaryKey, $this->primaryKey);
            if (!empty($optionIds)) {
                connect('car_options_value')->in($this->primaryKey, $optionIds)->delete();
                connect('car_option_models')->in($this->primaryKey, $optionIds)->delete();
            }

            return true;
        }

        return false;
    }


    /**
     * Сортировка
     *
     * @param $featureIds
     */
    public function reorder($featureIds)
    {
        $i = 0;
        foreach ($featureIds as $featureId) {
            $i++;
            connect($this->table)->where($this->primaryKey, $featureId)->update(['orderBy' => $i]);
        }
    }


    /**
     * Получение ids комплектаций
     * car_options_value
     *
     * @param int $featureId
     * @return array `$equipmentIds`
     */
    public function getFeatureEquipmentsIds(int $featureId): array
    {
        $equipmentIds = [];
        if ($equipments = connect()->table('car_options_value')->select('equipmentId')->where($this->primaryKey, $featureId)->keyBy('equipmentId')) {
            foreach ($equipments as $id => $val) $equipmentIds[$id] = $id;

        }
        return $equipmentIds;

    }


    public function updateOptionModels()
    {
        connect('car_option_models')->where('optionId', $this->id)->delete();


        if (is_array($this->modelIds)) {
            foreach($this->modelIds as $modelId)
                connect('car_option_models')->insert(['optionId' => $this->id, 'modelId' => $modelId]);
        }
    }

    public function getSelectOptionModels()
    {
        if (!empty($this->id)) {
            $this->modelIds = connect('car_option_models')->select('modelId')->where('optionId', $this->id)->pluck('modelId', 'modelId');
        }
    }

    public function getOptionModels($optionId)
    {
        $result = [];
        if (!empty($optionId)) {
           $result = connect('car_option_models AS com')
               ->leftJoin('car_models AS cm', 'cm.modelId', '=','com.modelId')
               ->select('cm.modelId, cm.title')
               ->where('com.optionId', $optionId)->getAll('array');
           if (!empty($result)) $result = array_column($result, 'title', 'modelId');
        }

        return $result;
    }

    /** OPTIONS */

    /**
     * Функция возвращает варианты товара
     * @return array|null
     */
    public function options(): ?array
    {
        if (!empty($this->id)) {
            return connect('car_options')->where('parentId', $this->id)->orderBy('orderBy')->indexKey('optionId')->keyBy('optionId');
        }

        return [];
    }

    public function option(int $optionId)
    {
        if (!empty($optionId)) {
            return connect('car_options')->where('optionId', $optionId)->get('array');
        }

        return null;
    }

    public function addOption(array $option): int
    {
        $option['parentId'] = $this->id;
        $option['orderBy'] = 0;

        return connect('car_options')->insert($option);
    }

    /**
     * @param int $optionId
     * @param array $option
     */
    public function updateOption(int $optionId, array $option): void
    {
        connect('car_options')->where('optionId', $optionId)->update($option);
    }

    public function removeOption(int $optionId): void
    {
        connect('car_options')->where('optionId', $optionId)->delete();
    }


}