<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Http\Request;
use Mnv\Models\Exceptions\NoContentException;
use Mnv\Models\Exceptions\NotFoundException;

/**
 * Class ContentTypeCustomField
 * @package Mnv\Models
 */
class ContentTypeCustomField extends Model
{
    /** @var string */
    protected string $table = 'type_content_custom_fields';
    protected string $table_custom_values = 'type_content_custom_values';
    protected string $table_field_property = 'type_content_field_property';

    /** @var string  */
    protected string $primaryKey = 'fieldId';

    /** @var string  */
    protected string $orderBy = 'orderBy ASC';

    /** @var int|null  */
    public $typeId;

    /** @var array|mixed  */
    public $field_data = [];

    /** @var  */
    public $properties;

    public $fieldIds = [];

    /** @var array|string[]  */
    private static array $typeFields = [
        'I' => 'Поле ввода',        // I - text
        'D' => 'Дата',              // D - data
        'C' => 'Флажок',            // C - checkbox
        'R' => 'Радиогруппа',       // R - radio
        'S' => 'Список вариантов',  // S - select
        'T' => 'Текстовая область', // T - textarea
//        'M' => 'Список вариантов мульти',  // M - select multiple
//        'F' => 'Файл'        // F - file
    ];


    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->data         = $request->get('field');
        $this->typeId       = $request->get('typeId');
        $this->properties   = $request->get('properties');
        $this->fieldIds     = $request->get('ids');

    }


    /**
     * Проверка на совпадение и получение fileName
     *
     * @param string|null $fileName
     * @return mixed|string|null
     */
    public function checkFileName(string $fileName)
    {
        if (!empty($this->id)) connect()->where($this->primaryKey, '<>', $this->id);
        if ($fileName = connect()->table($this->table)->select('fileName')->where('typeId', $this->typeId)->where('LOWER(fileName)', strtolower($fileName))->getValue()) {
            return $fileName;
        }

        return null;
    }

    public function prepare(array $data, int $managerId): bool
    {
        // Установите детали модификации
        $currentTimestamp = gmdate('Y-m-d H:i:s');
        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = $currentTimestamp;

        if (empty($this->id)) {
            // Задайте сведения о создании новых записей
            $data['addedBy']       = $managerId;
            $data['addedOn']       = $currentTimestamp;
            $data['orderBy']       = $this->getMaxValue('orderBy') + 1;

            $this->id = $this->insert($data);
        } else {
            // Обновить существующую запись
            // Update existing record
            if (!$this->update($data)) {
                return false; // Return early on failure to avoid unnecessary operations
            }
        }

        // Если запись была успешно вставлена/обновлена
        if (isset($data['type']) && in_array($data['type'], ['R', 'S', 'C'], true)) {
            // Действуйте только в том случае, если fileName и type_sql существуют, чтобы избежать ненужных вызовов
            if (!empty($data['fileName']) && !empty($data['type_sql'])) {
                $this->createColumn($data['fileName'], $data['type_sql']);
                $this->prepareProperties();
            }
        }

        return (bool)$this->id;
    }


    /**
     * Удаление
     *
     * @return bool
     * @throws NotFoundException
     * @throws NoContentException
     */
    public function remove(): bool
    {
        // Проверьте, доступен ли идентификатор поля
        if (empty($this->id)) {
            throw new NotFoundException("Not Found: ID is required.");
        }

        // Выборка поля по первичному ключу `fieldId`
        $currentRecord = connect($this->table)->where($this->primaryKey, $this->id)->get('array');

        // Throw exception if no record is found for the given `id`
        if (!$currentRecord) {
            throw new NoContentException("No Content: Record not found for the given ID.");
        }

        // Удалите поле и связанные с ним пользовательские значения
        $this->deleteField();
        // Удалить пользовательские значения
        $this->deleteCustomValues($currentRecord['fileName']);

        // удаление поля в контенте
        if (in_array($currentRecord['type'], ['R', 'S', 'C'])) {
            $this->removeColumn($currentRecord['fileName']);
            // очистить свойства
            $this->clearProperties($this->id);
        }

        return true;
    }

    /**
     * Удалить поле
     *
     * @return void
     */
    private function deleteField(): void
    {
        connect()->table($this->table)->where($this->primaryKey, $this->id)->delete();
    }

    /**
     * Удалить пользовательские значения
     *
     * @param $fileName
     *
     * @return void
     */
    private function deleteCustomValues($fileName): void
    {
        connect()->table($this->table_custom_values)->where('fileName', $fileName)->delete();
    }

    /** ************************ ARTICLES ************************* */

    protected function createColumn($columnName, $sql = null): bool
    {
        $Existence_column = connect('articles')->isColumn($columnName)->fetch();
        if (!$Existence_column) {
            connect('articles')->createColumn($columnName, $sql)->exec();
            return true;
//         print_r(connect()->getQuery().PHP_EOL);
        }

        return false;
    }

    protected function removeColumn($columnName): bool
    {
        $Existence_column = connect('articles')->isColumn($columnName)->fetch();
        if ($Existence_column) {
           connect('articles')->removeColumn($columnName)->exec();
           return true;
        }
        return false;
    }


    /** ************************ PROPERTIES ************************* */

    /**
     * Получить все своства
     *
     * @return void
     */
    public function getProperties(): void
    {
        if (!empty($this->id)) {
            $this->properties = connect($this->table_field_property)->orderBy('propertyId ASC')->where($this->primaryKey, $this->id)->getAll('array');
        }
    }

    /**
     * Обработать все свойства
     *
     * @return void
     */
    protected function prepareProperties(): void
    {
        // Досрочный возврат, если свойства не заданы или это не массив
        if (empty($this->properties) || !is_array($this->properties)) {
            // очистить свойства
            $this->clearProperties($this->id);
            return;
        }

        $updatedPropertyIds = [];

        foreach ($this->properties as &$property) {
            $property['fieldId'] = $this->id;

            if (!empty($property['propertyId'])) {
                // Обновить существующее свойство
                $this->updateProperty($property['propertyId'], $property);
            } else {
                // Добавить новое свойство
                $property['propertyId'] = $this->addProperty($property);
            }

            // Извлеките и проверьте обновленное свойство
            if ($fetchedProperty = $this->getProperty($property['propertyId'])) {
                $updatedPropertyIds[] = $fetchedProperty['propertyId'];
            }
        }

        // Получить существующие свойства для сравнения
        $existingPropertyIds = $this->getListProperties($this->id);

        // Удаление свойств, которые больше не существуют
        $propertiesToDelete = array_diff($existingPropertyIds, $updatedPropertyIds);
        foreach ($propertiesToDelete as $propertyId) {
            $this->deleteProperty($propertyId);
        }

        // Сортировка и обновление порядка следования свойств
        foreach (array_values($updatedPropertyIds) as $index => $propertyId) {
            $this->updateProperty($propertyId, ['orderBy' => $index]);
        }
    }

    /**
     * Получить все ID свойтв данного поля
     *
     * @param int $fieldId
     *
     * @return array|null
     */
    protected function getListProperties(int $fieldId): ?array
    {
        if (!empty($fieldId)) {
            return connect($this->table_field_property)->where($this->primaryKey, $fieldId)->orderBy('propertyKey ASC')->pluck('propertyId', 'propertyId');
        }

        return null;
    }

    /**
     * Получить свойство
     *
     * @param int $propertyId  ID свойтва
     *
     * @return array|null]
     */
    protected function getProperty(int $propertyId): ?array
    {
        return !empty($propertyId)
            ? connect($this->table_field_property)->where('propertyId', $propertyId)->get('array')
            : null;
    }

    /**
     * Добавить свойство
     *
     * @param array $property  свойтво
     *
     * @return int
     */
    protected function addProperty(array $property): int
    {
        return (int)connect()->table($this->table_field_property)->insert($property);
    }

    /**
     * Обновить свойство
     *
     * @param int $propertyId ID  свойтва
     * @param array $property  свойтво
     *
     * @return void
     */
    protected function updateProperty(int $propertyId, array $property): void
    {
        connect($this->table_field_property)->where('propertyId', $propertyId)->update($property);
    }

    /**
     * Удалить свойтво
     *
     * @param int $propertyId ID свойтва
     *
     * @return void
     */
    protected function deleteProperty(int $propertyId): void
    {
        connect($this->table_field_property)->where('propertyId', $propertyId)->delete();
    }

    /**
     * Удалить все свойтва данного поля
     * @param  int  $fieldId поле
     *
     * @return void
     */
    protected function clearProperties(int $fieldId): void
    {
        connect($this->table_field_property)->where($this->primaryKey, $fieldId)->delete();
    }


    /** STATIC METHODS */


    public static function getTypeFields(): array
    {
        return self::$typeFields;
    }

    public static function getNameTypeField($type): string
    {
        return self::$typeFields[$type];
    }

    /**
     * @param int $typeId
     * @param string|null $status
     * @return array|null
     */
    public static function contentTypeFields(int $typeId, ?string $status): ?array
    {
        if (!empty($status)) connect()->where('status', $status);

        return connect('type_content_custom_fields')->where('typeId', $typeId)->orderBy('orderBy ASC')->getAll('array');
    }


    /**
     * Retrieve content type field property.
     *
     * @param string $orderBy The column to order the results by.
     * @param int $fieldId The identifier of the field.
     * @return array|null The resulting properties or null if none found.
     */
    public static function contentTypeFieldProperty(string $orderBy, int $fieldId): ?array
    {
        return connect('type_content_field_property')->orderBy($orderBy)->where('fieldId', $fieldId)->pluck('name', 'propertyKey');
    }

}