<?php

namespace Mnv\Models;

use InvalidArgumentException;
use Mnv\Core\Model;
use Mnv\Http\Request;

/**
 * Class OrderStatus
 * @package Mnv\Models
 */
class Deal extends Model
{
    /** @var string */
    protected string $table = 'car_deal_types';

    /** @var string  */
    protected string $primaryKey = 'id';

    protected string $orderBy = 'id ASC';

    public function __construct(Request $request)
    {
        $this->data = $request->get('deal');
    }


    /**
     * @return bool
     */
    /**
     * Prepares the data by synchronizing the current deals with the new data.
     *
     * - Fetches all existing records from the table.
     * - Indexes existing deals by their ID for quick access.
     * - Processes new data, updating, inserting, or deleting records as necessary.
     * - Throws an exception for deals with empty 'name' or 'fileName'.
     *
     * @return bool Returns true after successful synchronization of the data.
     */
    public function prepare(): bool
    {
        // Получаем все текущие записи из таблицы
        $existingDeals = connect($this->table)->getAll('array');

        // Подготовка массива существующих сделок для быстрого доступа
        $existingDealsIndexedById = [];
        foreach ($existingDeals as $deal) {
            $existingDealsIndexedById[$deal['id']] = $deal;
        }

        // Обрабатываем новые данные
        foreach ($this->data as $deal) {
            // Пропускаем сделки с пустым полем name или fileName
            if (empty($deal['name']) || empty($deal['fileName'])) {
                if (!empty($deal['id']) && isset($existingDealsIndexedById[$deal['id']])) {
                    connect($this->table)->where($this->primaryKey, $deal['id'])->delete();
                    unset($existingDealsIndexedById[$deal['id']]);
                }
                throw new InvalidArgumentException("Ошибка в наполнение, поля не должны быть пустыми.");
            }

            if (!empty($deal['id']) && isset($existingDealsIndexedById[$deal['id']])) {
                // Обновляем существующую запись
                connect($this->table)->where($this->primaryKey, $deal['id'])
                    ->update([
                        'name' => $deal['name'],
                        'fileName' => $deal['fileName']
                    ]);
                // Удаляем из списка существующих записей
                unset($existingDealsIndexedById[$deal['id']]);
            } else {
                // Вставляем новую запись
                connect($this->table)->insert([
                    'name' => $deal['name'],
                    'fileName' => $deal['fileName']
                ]);
            }
        }

        // Удаляем записи, которых больше нет в новых данных
        foreach ($existingDealsIndexedById as $oldDeal) {
            connect($this->table)->where($this->primaryKey, $oldDeal['id'])->delete();
        }

        return true;
    }


    public static function getDealTypes()
    {
        return connect('car_deal_types')->select('name, fileName')->orderBy('id ASC')->getAll('array');
    }

}