<?php

namespace Mnv\Models;

use Mnv\Core\Model;
use Mnv\Core\Test\Logger;
use Mnv\Http\Request;

/**
 * Class UserGroups
 * @package Mnv\Models\Users
 */
class UserGroups extends Model
{
    const ADMIN         = 'A';  // Администраторы
    const DEVELOPER     = 'D';  // Разработчики
    const MANAGER       = 'M';  // Менеджеры
    const MODERATOR     = 'O'; // Модератор
    const EDITOR        = 'E'; // Редактор
    const CONSULTANT    = 'K'; // Консультант
    const CUSTOMER      = 'C'; // Пользователи
    const SUBSCRIBER    = 'S'; // Подписчики


    /** @var string */
    protected string $table = 'user_group';
    protected string $table_privileges = 'user_group_privileges';

    /** @var string  */
    protected string $primaryKey = 'groupId';

    /** @var string */
    protected string $orderBy = 'groupId ASC';

    /** @var string */
    protected string $columns = 'groupId, groupName, status, addedOn, modifiedOn';

    /** @var array|mixed  */
    public $privileges = array();

    /** @var array|string[]  */
    public array $groups = array(
        self::DEVELOPER     => 'Разработчики',
        self::ADMIN         => 'Администраторы',
        self::MANAGER       => 'Менеджеры',
        self::MODERATOR     => 'Модератор',
        self::EDITOR        => 'Редактор',
        self::CONSULTANT    => 'Консультант',
        self::CUSTOMER      => 'Пользователи',
        self::SUBSCRIBER    => 'Подписчики',
    );

    public function __construct(Request $request)
    {
        $this->id           = $request->get('id');
        $this->data         = $request->get('group');
        $this->privileges   = $request->get('privileges');
    }


    /**
     * @param $limit
     * @param $page
     * @param $userGroups
     *
     * @return array|mixed
     */
    public function all($limit, $page, $userGroups = null)
    {
        if ($userGroups != 'D') connect()->notWhere($this->primaryKey, 1);

        return parent::all($limit, $page);
    }

    public function total($UserType = null): void
    {
        if ($UserType != 'D') connect()->notWhere($this->primaryKey, 1);

        parent::total();
    }

    /**
     * Получить группу
     */
    public function edit(): UserGroups
    {
        if (empty($this->id)) {
            return $this;
        }

        if ($this->data = connect($this->table)->where($this->primaryKey, $this->id)->get('array')) {
            $this->data['countUsers'] = $this->getCountGroupUsers($this->id);
            $this->data['privilege'] = $this->getUserPrivileges();
        }

        return $this;
    }

    /**
     * Получить кол-во пользователей группы
     *
     * @param int $groupId
     *
     * @return int
     */
    public function getCountGroupUsers(int $groupId): int
    {
        return (int) connect('users')->count('*', 'count')->where('groupId', $groupId)->getValue() ?? 0;
    }

    private function getUserPrivileges(): ?array
    {
        return connect($this->table_privileges)->select('privilege')->where($this->primaryKey, $this->id)->pluck('privilege');
    }


    /**
     * Проверить существует группа или нет с таким `$fileName`
     *
     * @param string|null $fileName
     * @return mixed|string|null
     */
    public function checkFileName(?string $fileName): ?string
    {
        if (empty($fileName)) {
            return null;
        }

        if (!empty($this->id)) {
            connect()->where($this->primaryKey,'<>', $this->id);
        }

        $existingFileName = connect($this->table)->select('fileName')->where('LOWER(fileName)', strtolower($fileName))->getValue();

        return $existingFileName ?: null;
    }


    /**
     * @param array $data
     * @param int $managerId
     *
     * @return bool
     */
    public function prepare(array $data, int $managerId): bool
    {
        $data['modifiedBy']   = $managerId;
        $data['modifiedOn']   = gmdate('Y-m-d H:i:s');

        if (empty($this->id)) {

            $data['addedBy'] = $managerId;
            $data['addedOn'] = gmdate('Y-m-d H:i:s');
            if ($this->id = $this->insert($data)) {
                $this->privileges();
                Logger::init()->info("Создана новая группа «" . $data['groupName'] . "»", $managerId)->save();
                return true;
            }
        } else {
            if ($this->update($data)) {
                $this->privileges();
                Logger::init()->info("Группа  «" . $data['groupName'] . "» была отредактирована", $managerId)->save();
                return true;
            }
        }

        return false;
    }


    /**
     * Удаление
     *
     * @return bool
     *
     * @throws Exceptions\NoContentException
     * @throws Exceptions\NotFoundException
     * @throws \Mnv\Core\Database\Throwable\DatabaseException
     */
    public function remove(): bool
    {
        if (parent::remove()) {
            connect($this->table_privileges)->where($this->primaryKey, $this->id)->delete();

            $userIds = connect('users')->where('groupId', $this->id)->pluck('userId', 'userId');
            if (!empty($userIds)) {
                connect('users')->in('userId', $userIds)->delete();
                connect('user_images')->in('userId', $userIds)->delete();
            }

            return true;
        }

        return false;
    }

    /**
     * Сохранение доступа к пунктам меню (привелегии)
     * @return void
     */
    private function privileges(): void
    {
        connect($this->table_privileges)->where($this->primaryKey, $this->id)->delete();

        foreach ($this->privileges as $privilege) {
            connect($this->table_privileges)->insert([
                'groupId'   => $this->id,
                'privilege' => $privilege
            ]);
        }
    }



    /** STATIC  user_group */

    public static function selectGroup($userType): array
    {
        $group = ['0' => "Выберите группу..."];
        if ($userType != 'D') connect()->notWhere('groupId', 1);
        $group += connect('user_group')->select('groupId, groupName')->where('status', 'V')->pluck('groupName','groupId');

        return $group;
    }

    public static function arrayGroup(): array
    {
        return connect('user_group')->select('groupId')->where('status', 'V')->pluck('groupId', 'groupId');
    }

    /**
     * @param $groupId
     * @return string|null
     */
    public static function groupName($groupId): ?string
    {
        return connect('user_group')->select('groupName')->where('groupId', $groupId)->getValue();
    }
}