<?php


namespace Mnv\Modules\Cars;


use Mnv\Core\Uploads\ImageCollection;

/**
 * Class Model
 * @package Mnv\Modules\Cars
 */
class PropertiesPage
{
    /** @var int */
    private int $modelId;

    /** @var array  */
    public array $model = [];

    public function __construct($modelId)
    {
        $this->modelId = $modelId;
    }

    /** получить минимальную цену комплектаций */
    public function getMinPriceEquipments(): PropertiesPage
    {
        $this->model['min_price'] = connect('car_equipments')->select('price')->where('status', 'V')->where('modelId', $this->modelId)->min('price', 'min_price')->getValue();

        return $this;
    }

    /**
     * Получить комплектации по модели
     */
    public function getEquipments(): PropertiesPage
    {
        $equipments = connect('car_equipments')->select('title, price, oldPrice, equipmentId, modificationId')->where('status', 'V')->where('modelId', $this->modelId)->orderBy('publishedOn ASC')->getAll('array');
        $this->model['equipments'] = collect($equipments)->mapWithKeys(function ($item) {
            $item['modification']   = connect('car_modifications')->select('modificationId, name, power, engine, transmissions, gear')->where('modificationId', $item['modificationId'])->get('array');
            $item['specifications'] = $this->getSpecifications($item['modificationId']);

            return [$item['equipmentId'] => $item];
        })->all();

        return $this;
    }

    /**
     * Спецификация
     * @param $modificationId
     * @return array|null
     */
    private function getSpecifications($modificationId): ?array
    {
        $specifications = connect('car_specifications')->select('specificationId, parentId, name')->orderBy('orderBy ASC')->where('status', 'V')->getAll('array');
        $specifications = $this->buildTreeFromArray($specifications, 'specificationId', 'property');
        return collect($specifications)->mapWithKeys(function ($specification) use ($modificationId) {
             $specification['property'] = collect($specification['property'])->mapWithKeys(function ($item) use ($modificationId) {
                $property = connect()->table('car_specifications_value')->select('value, unit')->where('specificationId', $item['specificationId'])->where('modificationId', $modificationId)->get('array');
                if (!empty($property['value'])) {
                    return [$item['specificationId'] => $property];
                }
                return [];
            })->all();

            return [$specification['specificationId'] => $specification];

        })->all();
    }

    /** получить стандартные опции  `hardware` == `S` */
    public function standardOptions(): PropertiesPage
    {
        $standards = connect('car_options')->select('optionId, parentId, name')->orderBy('orderBy ASC')->where('hardware', 'S')->where('status', 'V')->getAll('array');
        $standards = $this->buildTreeFromArray($standards, 'optionId', 'options');
        $this->model['standard_options'] = collect($standards)->map(function ($standard) {
            if (!empty($standard['options'])) {
                $standard['options'] = collect($standard['options'])->mapWithKeys(function ($item) {
                    return [$item['optionId'] => $item['name']];
                })->all();

                unset($standard['parentId']);
                unset($standard['optionId']);
                if (!empty($standard['options'])) {
                    return $standard;
                }
            }
            return null;
        })->all();

        return $this;
    }

    /** получить все спецификации */
    public function specifications(): PropertiesPage
    {
        $specifications = connect('car_specifications')->select('specificationId, parentId, name')->orderBy('orderBy ASC')->where('status', 'V')->getAll('array');
        $specifications = $this->buildTreeFromArray($specifications, 'specificationId', 'property');

        $newSpecifications = collect($specifications)->mapWithKeys(function ($specification) {
            $specification['property'] = collect($specification['property'])->mapWithKeys(function ($item) {
                return [$item['specificationId'] => $item['name']];
            })->all();

            unset($specification['parentId']);
            return [$specification['specificationId'] => $specification];
        })->all();

        $this->model['specifications'] = $newSpecifications;

        return $this;
    }


    /** Цвет кузова */
    public function getModelBColors(): PropertiesPage
    {
        $color = connect()->table('car_model_colors AS cec')->usingJoin('car_colors AS cc', 'colorId')
            ->select('cc.colorName AS name, cc.colorCode AS code, cec.fileId, cec.price, cec.menu')
            ->where('cec.type', 'bcolors')->where('cec.menu', 'Y')
            ->where('cec.modelId', $this->modelId)->orderBy('cec.orderBy ASC')->get('array');

        if ($file = connect('files')->where('fileId', $color['fileId'])->get('array')) {
            $color = ImageCollection::init()->get(null, $file);
        }
        $this->model['image'] = $color;

        return $this;
    }


    /** РАБОТА С ФАЙЛАМИ / ИЗОБРАЖЕНИЕМ */

    /** получить все из таблицы images модели  */
    public function getFiles($table, $primaryKey, $sectionId): ?array
    {
        $files = [];
        if ($images = connect($table)->where($primaryKey, $sectionId)->select('fileId, type, position')->orderBy('orderBy ASC')->getAll('array')) {
            collect($images)->map(function ($item) use (&$files) {
                if ($file = connect('files')->where('fileId', $item['fileId'])->get('array')) {
                    if ($item['type'] == 'general') {
                        $files['general'] = ImageCollection::init()->get($item, $file);
                    } elseif ($item['type'] == 'gallery') {
                        $files['gallery'][] = ImageCollection::init()->get($item, $file);
                    }
                    else {
                        $files['docs'][] = ImageCollection::init()->get($item, $file);
                    }
                }

                return $files;
            })->toArray();
        }

        return $files;
    }




    /** работа с массивом  */
    private function buildTreeFromArray($items, $id, $subArray)
    {
        $childs = [];

        foreach ($items as &$item) {
            $childs[$item['parentId'] ?? 0][] = &$item;
        }

        unset($item);

        foreach ($items as &$item) {
            if (isset($childs[$item[$id]])) {
                $item[$subArray] = $childs[$item[$id]];
            }
        }

        return $childs[0] ?? [];
    }

}