<?php

namespace Mnv\Modules;

use Mnv\Core\Locale\I18N;
use Mnv\Core\Utilities\Cookie\Cookie;

class Favorites
{

    /** @var array */
    protected array $list = [];

    /** @var int */
    protected int $userId = 0;

    /** @var bool */
    protected bool $allowGuest;

    /** @var int */
    protected int $limit;

    /** @var int */
    protected int $id = 0;

    /** @var int */
    public int $countFavorite = 0;

    /** @var string  */
    public string $error;

    /** @var string  */
    public string $message;

    /** @var string  */
    public string $button;

    /** @var string  */
    public string $date;

    /** @var bool  */
    public bool $isRemoved = false;

    /** Messages **/
    protected const ERROR_NO_ID = 'Не задан ID';
    protected const ERROR_GUEST = 'Гости не могут добавлять в избранное, необходимо авторизоваться на сайте';

    /**
     * Конструктор класса
     */
    public function __construct($userId, bool $allowGuest = false, int $limit = 100)
    {
        $this->userId      = $userId;
        $this->allowGuest  = $allowGuest;
        $this->limit       = $limit;
    }
    public function setId($id): bool
    {
        if ($id < 1) {
            $this->message = self::ERROR_NO_ID;
            return false;
        }

        if (!$this->userId && !$this->allowGuest) {
            $this->message = self::ERROR_GUEST;
            return false;
        }

        $this->loadList();
        $this->toggleFavorite($id);
        return $this->syncFavorites();
    }

    protected function toggleFavorite($id): void
    {
        if (isset($this->list[$id])) {
            unset($this->list[$id]);
            $this->isRemoved = true;
        } else {
            $this->list = [$id => true] + $this->list;
            $this->isRemoved = false;
        }

        $this->applyLimit();
        $this->countFavorite = count($this->list);
    }

    protected function applyLimit(): void
    {
        if ($this->limit > 0 && count($this->list) > $this->limit) {
            array_pop($this->list);
        }
    }

    protected function syncFavorites(): bool
    {
        $list = array_keys($this->list);
        $this->syncToCookie($list);

        if ($this->isRemoved) {
            $this->message =  I18N::locale("Товар был удален из списка избранных...", "Mahsulot Sevimlilar ro'yxatidan olib tashlandi...", "The product has been removed from favorites...");
            $this->button =  I18N::locale("Добавить в избранное", "Tanlanganlarga qo'shish", "Add to favorites");
        } else {
            $this->message =  I18N::locale("Товар был добавлен в список избранных...", "Mahsulot Sevimlilar ro'yxatiga qo'shildi...", "The product has been added to the favorites list...");
            $this->button =  I18N::locale("Удалить из избранного", "Tanlanganlardan o'chirish", "Delete from favorites");
        }

        return true;
    }

    protected function loadList(): void
    {
        $list = $this->getGuestFavorites();
        $this->populateFavorites($list);
    }

    protected function populateFavorites(array $list): void
    {
        foreach ($list as $id) {
            if ($id > 0) {
                $this->list[$id] = true;
            }
        }

        $this->applyLimit();
    }



    protected function syncToCookie(array $list): void
    {

        Cookie::setcookie('_favorites', join(',', $list), time() + 3600 * 24 * 1000, "/");
    }


    protected function getGuestFavorites(): array
    {
        $cookieFavorites = Cookie::get('_favorites');
        return !empty($cookieFavorites) ? explode(',', $cookieFavorites) : [];
    }

}